<!---
LICENSE INFORMATION:

Copyright 2010, Joe Rinehart, Dan Wilson

Licensed under the Apache License, Version 2.0 (the "License"); you may not 
use this file except in compliance with the License. 

You may obtain a copy of the License at 

	http://www.apache.org/licenses/LICENSE-2.0 
	
Unless required by applicable law or agreed to in writing, software distributed
under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR 
CONDITIONS OF ANY KIND, either express or implied. See the License for the 
specific language governing permissions and limitations under the License.

VERSION INFORMATION:

This file is part of Model-Glue 'Gesture' 3.1 Maintenance Release Beta (3.1.261).

The version number in parentheses is in the format versionNumber.subversion.revisionNumber.
--->

<cfcomponent hint="I am a simple time-based content cache.  Sweeps cache based on the ""sweepInterval"" constructor argument, checking for sweep necessity on each write to the cache.">

<cffunction name="init" access="public" hint="Constructor">
	<cfargument name="sweepInterval" type="numeric" hint="Number of seconds to wait between cache sweeps."	/>
	<cfargument name="defaultTimeout" type="numeric" hint="Number of seconds an item should live in a cache unless explicitly stated."	/>
	
	<cfset variables._content = structNew() />
	<cfset variables._sweepInterval = arguments.sweepInterval />
	<cfset variables._defaultTimeout = arguments.defaultTimeout />
	<cfset variables._lastSweep = now() />
	<cfset variables._lockName = createUUID() />
	
	<cfreturn this />
</cffunction>

<cffunction name="put" access="public" hint="Puts content into the cache.">
	<cfargument name="key" type="string" hint="Key for the content." />
	<cfargument name="content" type="any" hint="The content to cache." />
	<cfargument name="timeout" type="numeric" hint="Seconds this item should live in the cache." default="#variables._defaultTimeout#" />

	<cfset arguments.created = now() />
	<cfset variables._content[arguments.key] = arguments />	
</cffunction>

<cffunction name="get" access="public" returntype="struct" hint="Gets content from the cache.  Returns struct of {success:true, content=""content""}.  If not found, success will be false.">
	<cfargument name="key" type="string" hint="Key for the content." />

	<cfset var content = structNew() />

	<cfif dateDiff("s", variables._lastSweep, now()) gt variables._sweepInterval>
		<cflock name="#variables._lockName#" type="exclusive" timeout="30">
			<cfif dateDiff("s", variables._lastSweep, now()) gt variables._sweepInterval>
				<cfset sweep() />
			</cfif>
		</cflock>
	</cfif>

	<cftry>
		<cfif structKeyExists(variables._content, arguments.key)>
			<cfset content.success = true />
			<cfset content.content = variables._content[arguments.key].content />
		<cfelse>
			<cfset content.success = false />
		</cfif>
		<cfcatch>
			<cfset content.success = false />
		</cfcatch>
	</cftry>

	<cfreturn content />
</cffunction>

<cffunction name="sweep" access="public" returntype="void" hint="Instructs implementation to sweep stale items.">
	
	<cfset var key = "" />
	<cfset var item = "" />
	<cfset var sinceLastSweep = dateDiff("s", variables._lastSweep, now()) />
	
	<cfloop collection="#variables._content#" item="key">
		<cfset item = variables._content[key] />
		
		<cfif sinceLastSweep gt item.timeout>
			<cfset purge(key) />
		</cfif>
	</cfloop>
	
	<cfset variables._lastSweep = now() />
</cffunction>

<cffunction name="purge" access="public" returntype="void" hint="Purges content from the cache.">
	<cfargument name="key" type="string" hint="Key for the content." />
	
	<cfset structDelete(variables._content, key) />
</cffunction>

<cffunction name="getContents" access="public" returntype="struct" hint="Gets information about cache contents.">
	<cfreturn variables._content />
</cffunction>

</cfcomponent>