<!---
#   Copyright 2005 Sean A Corfield http://corfield.org/
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
--->
<cfcomponent name="IRCBot" hint="I am an IRC bot.">
	<cffunction name="onIncomingMessage" returntype="struct" access="public" output="false"
		hint="I am called automatically by the Java gateway code.">
		<cfargument name="event" type="struct" required="true"
			 hint="I contain the event information." />
		<!---
	   		event.data:
	   			botName	- name of this bot
	   			channel	- channel on which this message was received
	   					- "" if this was a private message
	   			sender	- nick of person who sent the message
	   			login	- login?
	   			hostname	- hostname?
	   			message	- incoming message
	   		result:
	   			response	- MESSAGE | ACTION | SILENT
	   			target	- channel or user to send message back to
	   			message	- message / action to send
	   	--->
	   	<cfset var result = structNew() />
	   	<cfset var response = "SILENT" /> <!--- by default we don't respond --->
	   	<cfset var target = arguments.event.data.channel />
	   	<cfset var message = "" />
	   	<cfset var from = arguments.event.data.sender />
	   	<cfset var inbound = arguments.event.data.message />
	   	<cfset var self = arguments.event.data.botName />
	   	<cfset var selfNameLen = len(self) />
	   	<cfset var inLen = len(inbound) />
	   	<cfset var command = "" />
	   	<cfset var cmdArgs = "" />
	   	<cfset var delimiters = ",.:; " />
	   	<cfset var seenLog = getBotSeenLog(self) />
	   	<cfset var seenRecord = 0 />
	   	
	   	<!--- private message, respond to user if at all --->
	   	<cfif target is "">
	   		<cfset target = from />
	   		<!--- strip ! from private messages in case user forgets --->
	   		<cfif left(inbound,1) is "!">
				<cfset inbound = trim(right(inbound,len(inbound)-1)) />
			</cfif>
	   	<cfelse>
	   		<!--- record the last message from every user --->
	   		<cfset seenRecord = structNew() />
	   		<cfset seenRecord.when = now() />
	   		<cfset seenRecord.said = inbound />
	   		<cfset seenLog[from] = seenRecord />
	   		<!--- see if the message was for us  --->
	   		<cfif inLen gt selfNameLen and left(inbound,selfNameLen) eq self>
	   			<cfset inbound = trim(right(inbound,len(inbound)-selfNameLen-1)) />
	   		<cfelseif left(inbound,1) is "!">
	   			<!--- possible command with ! --->
	   			<cfset inbound = trim(right(inbound,len(inbound)-1)) />
	   		<cfelse>
	   			<!---
	   				if you want the bot to respond to general comments in the
	   				channel, then process inbound here and set response to
	   				"ACTION" or "MESSAGE" and set message to whatever you want
	   				the bot to do (action) or say (message)
	   				<cfif find("seancorfield",inbound)>
	   					<cfset response = "ACTION" />
	   					<cfset message = "is happy to here his author mentioned!" />
	   				</cfif>
	   			--->

	   			<!--- general message, ignore it --->
	   			<cfset inbound = "" />
	   		</cfif>
	   	</cfif>
	   	
	   	<cfif inbound is not "">
	   		<!--- pull out the "command" from the message --->
	   		<cfset command = listFirst(inbound,delimiters) />
	   		<!--- assume we will respond with a message --->
   			<cfset response = "MESSAGE" />
	   		<cfswitch expression="#command#">

	   		<!--- tell user what time it is --->
	   		<cfcase value="time">
	   			<cfset message = from & ", the time is " & timeformat(now(),"long") & " on " & dateformat(now(),"medium") & "." />
	   		</cfcase>
	   		
	   		<!--- give help in a private message --->
	   		<cfcase value="help">
				<cfset giveHelp(self,from) />
				<cfset message = from & ", I sent you help in a private message!" />
	   		</cfcase>

	   		<!--- silly example command: slap someone with a trout --->
	   		<cfcase value="trout">
	   			<cfset cmdArgs = listRest(inbound,delimiters) />
	   			<cfif cmdArgs is "">
		   			<cfset message = from & ", seen wha'?" />
		   		<cfelse>
		   			<cfset response = "ACTION" />
		   			<cfset message = "slaps #cmdArgs# with a large trout!" />
		   		</cfif>
		   	</cfcase>

	   		<!--- seen a user? --->
	   		<cfcase value="seen">
	   			<cfset cmdArgs = listRest(inbound,delimiters) />
	   			<cfif cmdArgs is "">
		   			<cfset message = from & ", seen wha'?" />
		   		<cfelse>
		   			<cfif structKeyExists(seenLog,cmdArgs)>
		   				<cfset message = cmdArgs & " was last seen on " & timeformat(seenLog[cmdArgs].when,"long") &
		   									" on " & dateformat(seenLog[cmdArgs].when,"medium") &
		   									" saying " & seenLog[cmdArgs].said />
		   			<cfelse>
		   				<cfset message = "Sorry " & from & ", I haven't seen " & cmdArgs & " lately." />
		   			</cfif>
	   			</cfif>
	   		</cfcase>
	   		
	   		<!--- cfdocs.org lookup of livedocs URL --->
	   		<cfcase value="tag,fn" delimiters=",">
				<cfset cmdArgs = listRest(inbound,delimiters) />
				<cfhttp url="http://cfdocs.org/#cmdArgs#" />
				<cfif refind("Current page: http:[^<]*<",cfhttp.FileContent)>
					<cfset message = from & ", " & cmdArgs & " can be found at " &
							rereplace(cfhttp.filecontent,".*Current page: (http:[^<]*).*","\1") />
				<cfelse>
					<cfset message = from & ", sorry but I don't know what #cmdArgs# is!" />
				</cfif>
			</cfcase>
			
	   		<!--- these are admin commands - triggered by IRC events --->
	   		<cfcase value="admin">
		   		<cfset cmdArgs = listRest(inbound,delimiters) />
		   		<!--- record the last message from every user --->
		   		<cfset seenRecord = structNew() />
		   		<cfset seenRecord.when = now() />
		   		<cfswitch expression="#cmdArgs#">
			   		<cfcase value="onjoin">
				   		<cfset seenRecord.said = "[#from# joined #target#]" />
				   		<cfset seenLog[from] = seenRecord />
					</cfcase>
			   		<cfcase value="onpart">
				   		<cfset seenRecord.said = "[#from# left #target#]" />
				   		<cfset seenLog[from] = seenRecord />
					</cfcase>
			   		<cfcase value="onquit">
				   		<cfset seenRecord.said = "[#from# quit the server]" />
				   		<cfset seenLog[from] = seenRecord />
					</cfcase>
			   		<cfcase value="onkick">
				   		<cfset seenRecord.said = "[#from# was kicked from #target#]" />
				   		<cfset seenLog[from] = seenRecord />
					</cfcase>
				</cfswitch>
			</cfcase>

	   		<cfdefaultcase>
	   			<cfset message = "Sorry " & from & ", I don't understand " & inbound />
	   		</cfdefaultcase>
	   		</cfswitch>
	   	</cfif>

		<!--- fill in the result structure --->	   	
		<cfset result.response = response />
		<cfset result.target = target />
		<cfset result.message = message />
		<cfreturn result />
	</cffunction>
	
	<cffunction name="getBotData" returntype="struct" access="private" output="false">
		<cfargument name="botName" type="string" required="true" />
		<cfset var serverKey = "ircbot_" & arguments.botName />
		<cfset var lockName = "server_" & serverKey />
		<cfif not structKeyExists(server,serverKey)>
			<cflock name="#lockName#" type="exclusive" timeout="30">
				<cfif not structKeyExists(server,serverKey)>
					<cfset server[serverKey] = structNew() />
					<cfset server[serverKey].name = arguments.botName />
					<cfset server[serverKey].key = serverKey />
					<cfset server[serverKey].lockName = lockName />
				</cfif>
			</cflock>
		</cfif>
		<cfreturn server["ircbot_" & arguments.botName] />
	</cffunction>
	
	<cffunction name="getBotSeenLog" returntype="struct" access="private" output="false">
		<cfargument name="botName" type="string" required="true" />
		<cfset var botData = getBotData(arguments.botName) />
		<cfif not structKeyExists(botData,"seenLog")>
			<cflock name="#botData.lockName#_seenLog" type="exclusive" timeout="30">
				<cfif not structKeyExists(botData,"seenLog")>
					<cfset botData.seenLog = structNew() />
				</cfif>
			</cflock>
		</cfif>
		<cfreturn botData.seenLog />
	</cffunction>
	
	<cffunction name="giveHelp" returntype="void" access="private" output="false">
		<cfargument name="botName" type="string" required="true" />
		<cfargument name="target" type="string" required="true" />
		
		<cfset var response = structNew() />
		<cfset var helpLine = "" />
		<!--- |-delimited list of messages to send to a user who requests help: --->
		<cfset var help = "" &
				"I am #arguments.botName#, the channel bot. You can invoke me by name or by using ! and I also watch " &
				"all the conversations so I can remember when people were last on channel (and what they said).|" &
				"I understand the following commands:|" &
				"help - displays this set of messages.|" &
				"time - displays the time and date.|" &
				"seen {nickname} - I'll tell you when I last saw {nickname} on channel and what they said.|" &
				"tag {cftagname} - I'll tell you the LiveDocs link for that tag (CFMX 7).|" &
				"fn {cffuncname} - I'll tell you the LiveDocs link for that function (CFMX 7).|" &
				"Examples:|" &
				"!seen seancorfield|" &
				"!tag cfapplication" />
		
		<cfset response.response = "MESSAGE" />
		<cfset response.target = arguments.target />
		
		<cfloop list="#help#" delimiters="|" index="helpLine">
			<cfset response.message = helpLine />
			<cfset GetGatewayHelper(arguments.botName).sendMessage(response) />
		</cfloop>
		
	</cffunction>
	
</cfcomponent>