/*
   Copyright 2007, Firemoss, LLC

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
package com.firemoss.modelglue {
	import com.firemoss.modelglue.eventhandler.EventHandler;
	import flash.utils.Dictionary;
	import com.firemoss.modelglue.error.ModelGlueError;
	import com.firemoss.modelglue.event.ModelGlueEvent;
	import com.firemoss.modelglue.util.Iterator;
	import com.firemoss.modelglue.eventhandler.MessageBroadcast;
	import mx.controls.Alert;
	import com.firemoss.modelglue.tags.Controller;
	
	/**
	 * This is the Model-Glue framework - you do not need to instantiate this class or work with it directly.
	 * 
	 * This class is the "moving part" of Model-Glue responsible for invoking event handlers.
	*/
	public class Framework {

		private static var instance:Framework;
		private var eventHandlers:Dictionary;
		private var controllers:Dictionary;
					
		public static function getInstance():Framework {
			if (!Framework.instance) Framework.instance = new Framework();
			
			return Framework.instance
		}
		
		public function Framework() {
			if (Framework.instance) throw new ModelGlueError(ModelGlueError.MODEL_GLUE_SINGLETON_EXCEPTION, "ModelGlue singleton error!");
			
			Framework.instance = this;

			this.eventHandlers = new Dictionary();
			this.controllers = new Dictionary();
		}
		
		public function addController(controller:Controller):void {
			this.controllers[controller.id] = controller;
		}
		
		public function getController(id:String):Controller {
			if (this.controllers[id] == null) {
				throw new ModelGlueError("Can't find controller \"" + id + "\"!", ModelGlueError.MODEL_GLUE_CONTROLLER_NOT_FOUND);
			}
			return this.controllers[id];
		}
		
		public function addMessageListener(messageName:String, object:Object, callback:Function):void {
			var mb:MessageBroadcast = MessageBroadcast.forMessageNamed(messageName);
			mb.addListener(object, callback);
		}
		
		public function addEventHandler(eh:EventHandler):void {
			this.eventHandlers[eh.name] = eh;
		}
		public function getEventHandlerFor(event:ModelGlueEvent):EventHandler {
			return getEventHandlerForEventNamed(event.name);
		}
		public function getEventHandlerForEventNamed(name:String):EventHandler {
			if (this.eventHandlers[name] == null) throw new ModelGlueError(ModelGlueError.MODEL_GLUE_EVENT_HANDLER_NOT_DEFINED, "No event handler named '" + name + "' exists!");	
			return this.eventHandlers[name];
		}
		
		public function handleEvent(e:ModelGlueEvent):void {
			var handler:EventHandler;
			
			while (e.hasNextEventHandler()) {
				handler = e.nextEventHandler();
				invokeEventHandlerForEvent(handler, e);	
			}
		}
		
		private function invokeEventHandlerForEvent(eh:EventHandler, e:ModelGlueEvent):void {
			var broadcasts:Array = eh.messageBroadcasts;
			var messageBroadcast:MessageBroadcast;
			var results:Array;
			var i:int;
			
			for (i=0;i<broadcasts.length;i++) {
				messageBroadcast = broadcasts[i] as MessageBroadcast;
				e.setCurrentMessageBroadcast(messageBroadcast);
				
				messageBroadcast.invokeListeners(e);
				
				if (e.stopped) {
					break;
				}
			}
			
			if (!e.stopped) {
				results = eh.getImplicitResults();
				
				for (i=0;i<results.length;i++) {
					e.addEventHandlerForEventNamed(results[i] as String);
				}
			}			
		}
	}
}